import logging
from aibot.tasks import do_business_logic
from aibot.utils import (
    build_bouquet_provider_list,
    build_cable_provider_list,
    build_plan_list,
    get_amount_value,
    get_cable_provider_list,
    get_plan_title,
    get_provider_list,
    get_provider_title,
    get_providers,
    send_whatsapp_message,
    sync_providers_and_plans,
)

logger = logging.getLogger(__name__)


def send_service_buttons(to, name):
    description = (
        "Welcome to Swiftlink!\n\n"
        "Swiftlink is your one-stop platform for digital services.\n"
        "You can buy Airtime, Data Bundles, and pay for DSTV subscriptions easily.\n\n"
        "Choose a service below to get started."
    )
    payload = {
        "type": "interactive",
        "interactive": {
            "type": "button",
            "body": {"text": f"{description}"},
            "action": {
                "buttons": [
                    {
                        "type": "reply",
                        "reply": {"id": "airtime_AIRTIME_FORM", "title": "Buy Airtime"},
                    },
                    {
                        "type": "reply",
                        "reply": {"id": "data_DATA_FORM", "title": "Buy Data"},
                    },
                    {
                        "type": "reply",
                        "reply": {"id": "dstv_DSTV_FORM", "title": "DSTV Subscription"},
                    },
                ]
            },
        },
    }
    send_whatsapp_message(to, payload)

    # WhatsApp allows max 3 buttons per message, so send support as a follow-up
    support_payload = {
        "type": "interactive",
        "interactive": {
            "type": "button",
            "body": {"text": "Need help? Reach out to our support team."},
            "action": {
                "buttons": [
                    {
                        "type": "reply",
                        "reply": {"id": "contact_support", "title": "Contact Support"},
                    },
                ]
            },
        },
    }
    return send_whatsapp_message(to, support_payload)


def send_airtime_flow(to: str, flow_token: str, mode: str = "published"):
    # Pre-fetch providers so the first screen has data immediately
    api_response = get_providers()
    if api_response and "data" in api_response:
        sync_providers_and_plans(api_response)
    providers = get_provider_list(category="Airtime") or [
        {"id": "0", "title": "No providers available"}
    ]
    payload = {
        "type": "interactive",
        "interactive": {
            "type": "flow",
            "header": {"type": "text", "text": "Buy Airtime"},
            "body": {"text": "Select network, enter phone and amount."},
            "action": {
                "name": "flow",
                "parameters": {
                    "flow_token": flow_token,
                    "flow_name": "buy_airtime_flow",  # or your actual flow_name/flow_id
                    # or your actual flow_name/flow_id
                    # "reply_id": "airtime_AIRTIME_FORM",
                    "flow_message_version": "3",
                    "flow_cta": "Buy Airtime",
                    "mode": mode,
                    # "flow_action": "navigate",
                    # "flow_action_payload": {
                    #     "screen": "DATA_FORM",
                    #     "data": {
                    #         "providers": providers,
                    #     },
                    # },
                },
            },
        },
    }
    return send_whatsapp_message(to, payload)


def send_data_flow(to: str, flow_token: str, mode: str = "published"):

    api_response = get_providers()
    if api_response and "data" in api_response:
        sync_providers_and_plans(api_response)
    providers = get_provider_list(category="Data") or [
        {"id": "0", "title": "No providers available"}
    ]
    payload = {
        "type": "interactive",
        "interactive": {
            "type": "flow",
            "header": {"type": "text", "text": "Buy Data Bundle"},
            "body": {"text": "Select network, enter phone and data plan."},
            "action": {
                "name": "flow",
                "parameters": {
                    "flow_token": flow_token,
                    "flow_name": "data_purchase_flow_new",
                    "flow_message_version": "3",
                    "flow_cta": "Buy Data Bundle",
                    "mode": mode,
                    "flow_action": "navigate",
                    "flow_action_payload": {
                        "screen": "DATA_FORM",
                        "data": {
                            "providers": providers,
                        },
                    },
                },
            },
        },
    }
    return send_whatsapp_message(to, payload)


def send_dstv_flow(to: str, flow_token: str, mode: str = "published"):
    # Pre-fetch cable providers so the first screen has data immediately
    providers = get_cable_provider_list() or [
        {"id": "0", "title": "No providers available"}
    ]
    payload = {
        "type": "interactive",
        "interactive": {
            "type": "flow",
            "header": {"type": "text", "text": "Buy DSTV Subscription"},
            "body": {"text": "Select network, enter phone and data plan."},
            "action": {
                "name": "flow",
                "parameters": {
                    "flow_token": flow_token,
                    "flow_name": "dstv_purchase_flow",  # or your actual flow_name/flow_id
                    "flow_message_version": "3",
                    "flow_cta": "Buy DSTV Subscription",
                    "mode": mode,
                    "flow_action": "navigate",
                    "flow_action_payload": {
                        "screen": "DSTV_FORM",
                        "data": {
                            "providers": providers,
                        },
                    },
                },
            },
        },
    }
    return send_whatsapp_message(to, payload)


FLOW_CONFIG = {
    # Airtime Flow
    "buy_airtime": {
        "flow_name": "buy_airtime_flow",
        "initial_screen": "AIRTIME_FORM",
        "confirm_screen": "CONFIRM_SCREEN",
        "complete_screen": "COMPLETE_SCREEN",
        "handler": "handle_buy_airtime",
    },
    # Example: Buy Data Flow
    "data_purchase": {
        "flow_name": "data_purchase_flow_new",
        "initial_screen": "DATA_FORM",
        "confirm_screen": "CONFIRM_SCREEN",
        "complete_screen": "COMPLETE_SCREEN",
        "handler": "handle_buy_data",
    },
    # # Example: Airtime → Cash Flow
    # "airtime_to_cash": {
    #     "flow_name":
    #     "initial_screen": "CASH_FORM",
    #     "confirm_screen": "CASH_SUMMARY",
    #     "complete_screen": "COMPLETE_SCREEN",
    #     "handler": "handle_airtime_to_cash",
    # },
    # DSTV Flow
    "dstv_purchase": {
        "flow_name": "dstv_purchase_flow",
        "initial_screen": "DSTV_FORM",
        "confirm_screen": "CONFIRM_SCREEN",
        "complete_screen": "COMPLETE_SCREEN",
        "handler": "handle_dstv_purchase",
    },
}


def handle_buy_airtime(form_data, flow_token, screen):

    def clean(value):
        if not value or value == "@@VALUE@@":
            return ""
        return value

    network = clean(form_data.get("network"))
    phone = clean(form_data.get("phone"))
    amount = clean(form_data.get("amount"))

    if screen == "AIRTIME_FORM" and not network:

        api_response = get_providers()
        if api_response and "data" in api_response:
            sync_providers_and_plans(api_response)

        providers = get_provider_list(category="Airtime") or [
            {"id": "loading", "title": "Loading..."}
        ]

        logger.info(f"[BUY_AIRTIME] {network=} {phone=} {amount=}")
        return {
            "screen": "AIRTIME_FORM",
            "data": {
                "network": providers,
                "phone": "",
                "amount": "",
            },
        }
    if screen == "AIRTIME_FORM" and network and  phone and amount:

        logger.info(f"[BUY_AIRTIME] {network=} {phone=} {amount=}")

        return {
            "screen": "CONFIRM_SCREEN",
            "data": {
                "confirm_network_text": f"Network: {network}",
                "confirm_phone_text": f"Phone: {phone}",
                "confirm_amount_text": f"Amount: ₦{amount}",
                "network": network,
                "phone": phone,
                "amount": amount,
            },
        }
    if screen == "CONFIRM_SCREEN":

        logger.info(f"[BUY_AIRTIME] {network=} {phone=} {amount=}")
        do_business_logic(form_data, "airtime")

        return {
            "screen": "COMPLETE_SCREEN",
            "data": {
               
                "confirm_message": f"You’re buying ₦{amount} worth of airtime for {phone} on {network}.",
                "network": network,
                "phone": phone,
                "amount": amount,
            },
        }

def handle_buy_data(form_data, flow_token, screen):

    def clean(value):
        if not value or value == "@@VALUE@@":
            return ""
        return value

    provider = clean(form_data.get("provider"))
    plan = clean(form_data.get("plan"))
    phone = clean(form_data.get("phone"))

    # 1️⃣ INIT → Providers only
    if screen == "DATA_FORM" and not provider:
        print("Loading providers for data purchase flow")
        # Sync providers from API to database
        api_response = get_providers()
        if api_response and "data" in api_response:
            sync_providers_and_plans(api_response)

        providers = get_provider_list(category="Data") or [
            {"id": "loading", "title": "Loading..."}
        ]

        return {
            "screen": "DATA_FORM",
            "data": {
                "providers": providers,
                "plans": [],
                "amount": "",
                "provider_title": "",
            },
        }

    # 2️⃣ Provider selected → Load plans
    if screen == "DATA_FORM" and provider and not plan:
        providers = get_provider_list(category="Data")
        plans = build_plan_list(provider) or [{"id": "none", "title": "No plans"}]

        return {
            "screen": "DATA_FORM",
            "data": {
                "providers": providers,
                "plans": plans,
                "plan": "",
                "amount": "",
                "provider_title": get_provider_title(provider),
            },
        }

    # 3️⃣ Plan selected → Fill amount
    if screen == "DATA_FORM" and plan and not phone:
        providers = get_provider_list(category="Data")
        plans = build_plan_list(provider)

        plan_info = get_plan_title(plan)

        return {
            "screen": "DATA_FORM",
            "data": {
                "providers": providers,
                "plans": plans,
                "amount": str(plan_info.amount),
                "provider_title": get_provider_title(provider),
            },
        }

    # 4️⃣ Continue → Confirm screen
    if screen == "DATA_FORM":
        plan_info = get_plan_title(plan)

        return {
            "screen": "CONFIRM_SCREEN",
            "data": {
                "provider": provider,
                "plan": plan,
                "phone": phone,
                "amount": str(plan_info.amount),
                "confirm_provider_text": f"Provider: {get_provider_title(provider)}",
                "confirm_plan_text": f"Plan: {plan_info.plan}",
                "confirm_phone_text": f"Phone: {phone}",
                "confirm_amount_text": f"Amount: ₦{plan_info.amount}",
            },
        }

    # 5️⃣ Pay now → Complete
    if screen == "CONFIRM_SCREEN":
        plan_info = get_plan_title(plan)
        do_business_logic(form_data, "data")

        return {
            "screen": "COMPLETE_SCREEN",
            "data": {
                "message": "Processing your purchase...",
                "provider": provider,
                "plan": plan,
                "phone": phone,
                "amount": str(plan_info.amount),
            },
        }


def handle_dstv_purchase(form_data, flow_token, screen):

    def clean(value):
        if not value or value == "@@VALUE@@":
            return ""
        return value

    provider = clean(form_data.get("provider"))
    bouquet = clean(form_data.get("bouquet"))
    smartcard = clean(form_data.get("smartcard"))
    phone = clean(form_data.get("phone"))

    print(
        f"[DSTV] screen={screen}, provider={provider}, bouquet={bouquet}, smartcard={smartcard}, phone={phone}"
    )

    # 1️⃣ INIT → Providers only
    if screen == "DSTV_FORM" and not provider:
        print("Loading providers for data purchase flow")
        # Sync providers from API to database
        api_response = get_providers()
        if api_response and "data" in api_response:
            sync_providers_and_plans(api_response)

        providers = get_cable_provider_list(category="Cable") or [
            {"id": "loading", "title": "Loading..."}
        ]

        return {
            "screen": "DSTV_FORM",
            "data": {
                "providers": providers,
                "bouquets": [],
                "amount": "",
                "provider_title": "",
            },
        }

    # 2️⃣ Provider selected → Load plans
    if screen == "DSTV_FORM" and provider and not bouquet:
        providers = get_cable_provider_list(category="Cable")
        print(f"[DSTV] Selected provider {provider}, loading bouquets")
        bouquets = build_cable_provider_list(provider) or [
            {"id": "none", "title": "No plans"}
        ]

        return {
            "screen": "DSTV_FORM",
            "data": {
                "providers": providers,
                "bouquets": bouquets,
                "bouquet": "",
                "amount": "",
                "provider_title": provider,
            },
        }

    # 3️⃣ Plan selected → Fill amount
    if screen == "DSTV_FORM" and bouquet and not smartcard and not phone:
        providers = get_cable_provider_list(category="Cable")
        bouquets = build_cable_provider_list(provider)

        bouquet_info = build_bouquet_provider_list(provider, bouquet)

        return {
            "screen": "DSTV_FORM",
            "data": {
                "providers": providers,
                "bouquets": bouquets,
                "amount": str(bouquet_info["variation_amount"]),
                "provider_title": provider,
            },
        }

    # 4️⃣ Continue → Confirm screen
    if screen == "DSTV_FORM":
        bouquet_info = build_bouquet_provider_list(provider, bouquet)

        return {
            "screen": "CONFIRM_SCREEN",
            "data": {
                "provider": provider,
                "bouquet": bouquet,
                "phone": phone,
                "smartcard": smartcard,
                "amount": str(bouquet_info["variation_amount"]),
                "confirm_provider_text": f"Provider: {provider}",
                "confirm_plan_text": f"Plan: {bouquet_info['name']}",
                "confirm_phone_text": f"Phone: {phone}",
                "confirm_amount_text": f"Amount: ₦{bouquet_info['variation_amount']}",
            },
        }

    # 5️⃣ Pay now → Complete
    if screen == "CONFIRM_SCREEN":
        bouquet_info = build_bouquet_provider_list(provider, bouquet)
        do_business_logic(form_data, "cable")

        return {
            "screen": "COMPLETE_SCREEN",
            "data": {
                "message": "Processing your purchase...",
                "provider": provider,
                "bouquet": bouquet,
                "phone": phone,
                "smartcard": smartcard,
                "amount": str(bouquet_info["variation_amount"]),
            },
        }


HANDLER_MAP = {
    "handle_buy_airtime": handle_buy_airtime,
    "handle_buy_data": handle_buy_data,
    "handle_dstv_purchase": handle_dstv_purchase,
}
