from django.db import models
from django.utils import timezone
import uuid


def gen_order_ref():
    return "ORD-" + uuid.uuid4().hex[:12].upper()


class Customer(models.Model):

    whatsapp_number = models.CharField(max_length=20, unique=True)
    name = models.CharField(max_length=120, blank=True, null=True)
    email = models.EmailField(blank=True, null=True)

    palmpay_account_number = models.CharField(max_length=20, blank=True, null=True)
    palmpay_bank_name = models.CharField(max_length=50, blank=True, null=True)
    palmpay_account_ref = models.CharField(max_length=100, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)

    # def save(self, *args, **kwargs):
    #     creating = self.pk is None
    #     super().save(*args, **kwargs)
    #     if creating and not self.palmpay_account_number:
    #         from aibot.services.palmpay import create_palmpay_virtual_account

    #         payload = {
    #             "requestTime": int(time() * 1000),
    #             "identityType": "company",
    #             "licenseNumber": "dasd141234114123",
    #             "virtualAccountName": "PPTV2",
    #             "version": "V2.0",
    #             "customerName": self.name or self.whatsapp_number,
    #             "email": self.email or f"{self.whatsapp_number}@example.com",
    #             "nonceStr": "8GagBq4oGahVZAD8PQgLFJdhGQxoS1gy",
    #         }

    #         acct = create_palmpay_virtual_account(payload)
    #         print("PalmPay virtual account response:", acct)
    #         if acct:
    #             self.palmpay_account_number = acct.get("virtualAccountNo")
    #             self.palmpay_bank_name = acct.get("bankName")
    #             self.palmpay_account_ref = acct.get("accountReference")
    #             super().save(
    #                 update_fields=[
    #                     "palmpay_account_number",
    #                     "palmpay_bank_name",
    #                     "palmpay_account_ref",
    #                 ]
    #             )

    def __str__(self):
        return self.whatsapp_number


class Order(models.Model):
    SERVICE_CHOICES = [
        ("airtime", "Airtime"),
        ("data", "Data"),
    ]

    STATUS_CHOICES = [
        ("PENDING", "Pending"),
        ("WAITING_PAYMENT", "Waiting for Payment"),
        ("PAID", "Paid"),
        ("FAILED", "Failed"),
        ("COMPLETED", "Completed"),
    ]

    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    customer = models.ForeignKey(
        Customer, on_delete=models.CASCADE, related_name="orders"
    )
    ref = models.CharField(max_length=255, blank=True, null=True)
    plan_id = models.TextField(blank=True, null=True)
    smartcard = models.CharField(max_length=50, blank=True, null=True)
    service_type = models.CharField(max_length=20, choices=SERVICE_CHOICES)
    network = models.CharField(max_length=20)
    payload = models.TextField(blank=True, null=True)
    phone_number = models.CharField(max_length=20)
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default="PENDING")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f"{self.service_type} | {self.phone_number} | {self.status}"


class Payment(models.Model):
    STATUS_CHOICES = [
        ("PENDING", "Pending"),
        ("SUCCESS", "Success"),
        ("FAILED", "Failed"),
    ]

    order = models.OneToOneField(
        Order, on_delete=models.CASCADE, related_name="payment"
    )
    account_reference = models.CharField(max_length=50, unique=True)
    bank_name = models.CharField(max_length=50, blank=True, null=True)
    account_number = models.CharField(max_length=20, blank=True, null=True)
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default="PENDING")
    transaction_reference = models.CharField(max_length=100, blank=True, null=True)
    date_paid = models.DateTimeField(blank=True, null=True)
    palmpay_transaction_id = models.CharField(max_length=100, blank=True, null=True)
    palmpay_account_number = models.CharField(max_length=20, blank=True, null=True)
    palmpay_bank_name = models.CharField(max_length=50, blank=True, null=True)
    palmpay_account_ref = models.CharField(max_length=100, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def mark_success(self, transaction_ref: str):
        self.status = "SUCCESS"
        self.transaction_reference = transaction_ref
        self.date_paid = timezone.now()
        self.save()
        # update order
        self.order.status = "PAID"
        self.order.save()

    def __str__(self):
        return f"{self.order.id} | {self.status}"


class Provider(models.Model):
    subcategory_id = models.IntegerField(unique=True)
    title = models.CharField(max_length=100)
    category = models.CharField(max_length=50)
    serviceID = models.CharField(max_length=100, blank=True, null=True)
    status = models.IntegerField(default=1)

    def __str__(self):
        return self.title


class Plan(models.Model):
    provider = models.ForeignKey(
        Provider, related_name="plans", on_delete=models.CASCADE
    )
    plan = models.CharField(max_length=100)
    amount = models.DecimalField(max_digits=10, decimal_places=2)

    def __str__(self):
        return f"{self.plan} ({self.amount})"
