# bot/tasks.py
import json
from celery import shared_task

from aibot.services.palmpay import create_order_virtual_account
from aibot.utils import create_unique_order_ref, get_airtime_providers, get_plan_title, get_provider_title, send_whatsapp_message
from .models import Customer, Order, Payment


@shared_task(bind=True, max_retries=3)
def send_airtime_task(self, order_id):
    from .models import Order
    order = Order.objects.select_related("customer", "payment").get(id=order_id)
    try:
        # Call Swiftlink API for Airtime subscription
        from aibot.utils import fulfill_swiftlink_request
        result = fulfill_swiftlink_request(order)
        if result and result.get("success") == True:
            order.status = "COMPLETED"
            order.save()
            send_whatsapp_message(order.customer.whatsapp_number,
                                  {"type": "text", "text": {"body": "🎉 Airtime sending successful!"}})
        else:
            order.status = "FAILED"
            order.save()
            send_whatsapp_message(order.customer.whatsapp_number,
                                  {"type": "text", "text": {"body": "❌ Airtime sending failed."}})
    except Exception as e:
        self.retry(exc=e, countdown=30)


@shared_task(bind=True, max_retries=3)
def send_data_task(self, order_id):
    from .models import Order
    order = Order.objects.select_related("customer", "payment").get(id=order_id)
    try:
        # Call Swiftlink API for data bundle purchase
        from aibot.utils import fulfill_swiftlink_request
        result = fulfill_swiftlink_request(order)
        if result and result.get("status") == "success":
            order.status = "COMPLETED"
            order.save()
            send_whatsapp_message(order.customer.whatsapp_number,
                                  {"type": "text", "text": {"body": "🎉 Data bundle sent successfully!"}})
        else:
            order.status = "FAILED"
            order.save()
            send_whatsapp_message(order.customer.whatsapp_number,
                                  {"type": "text", "text": {"body": "❌ Data bundle sending failed."}})
    except Exception as e:
        self.retry(exc=e, countdown=30)


@shared_task(bind=True, max_retries=3)
def send_dstv_task(self, order_id):
    from .models import Order
    order = Order.objects.select_related("customer", "payment").get(id=order_id)
    try:
        # Call Swiftlink API for DSTV subscription
        from aibot.utils import fulfill_swiftlink_request
        result = fulfill_swiftlink_request(order)
        if result and result.get("status") == "success":
            order.status = "COMPLETED"
            order.save()
            send_whatsapp_message(order.customer.whatsapp_number,
                                  {"type": "text", "text": {"body": "🎉 DSTV subscription successful!"}})
        else:
            order.status = "FAILED"
            order.save()
            send_whatsapp_message(order.customer.whatsapp_number,
                                  {"type": "text", "text": {"body": "❌ DSTV subscription failed."}})
    except Exception as e:
        self.retry(exc=e, countdown=30)


def do_business_logic(form_data, flow_type):
    phone = form_data.get("phone")
    amount = form_data.get("amount")
    print("Form data received in task:", form_data)
    customer, _ = Customer.objects.get_or_create(whatsapp_number=phone)
    order_ref = create_unique_order_ref()
    if flow_type == "airtime":
        network = form_data.get("network")
        print("Selected Network:", network)
        subcategory_id = get_airtime_providers(network)
        order = Order.objects.create(
            ref=order_ref,
            customer=customer,
            service_type="airtime",
            network=subcategory_id,
            phone_number=phone,
            amount=amount,
            payload=json.dumps(form_data),
        )
        payment = Payment.objects.create(order=order, amount=amount,
                              account_reference=f"ORD-{order.id.hex[:10].upper()}")
        order.status = "WAITING_PAYMENT"
        order.save()
        create_order_virtual_account(customer, order)  # Create PalmPay virtual account for this order
        # Send PalmPay account details to customer
        if customer.palmpay_account_number and customer.palmpay_bank_name:
            print(f"Sending payment instructions to {phone} for order {order.ref} with amount {amount}")
            send_whatsapp_message(
                customer.whatsapp_number,
                {
                    "type": "text",
                    "text": {
                        "body": f"Please pay ₦{amount} to:\n\nAcct: {customer.palmpay_account_number}\nBank: {customer.palmpay_bank_name}\nRef: {payment.order.ref}\n\nYour service will be delivered after payment is confirmed."
                    }
                }
            )

    elif flow_type == "data":
        provider_id = form_data.get("provider")
        plan_id = form_data.get("plan")
        phone = form_data.get("phone")

        provider = get_provider_title(provider_id)
        plan = get_plan_title(plan_id)
        order = Order.objects.create(
            ref=order_ref,
            customer=customer,
            service_type="data",
            network=provider_id,
            phone_number=phone,
            amount=amount,
            plan_id=plan_id,
            payload=json.dumps(form_data),
        )

        payment = Payment.objects.create(order=order, amount=amount,
                              account_reference=f"ORD-{order.id.hex[:10].upper()}")
        order.status = "WAITING_PAYMENT"
        order.save()
        create_order_virtual_account(customer, order)  # Create PalmPay virtual account for this order

        if customer.palmpay_account_number and customer.palmpay_bank_name:
            send_whatsapp_message(
                phone,
                {
                    "type": "text",
                    "text": {
                        "body": f"Please pay ₦{amount} to:\n\nAcct: {customer.palmpay_account_number}\nBank: {customer.palmpay_bank_name}\nRef: {payment.account_reference}\n\nYour service will be delivered after payment is confirmed."
                    }
                }
            )

    elif flow_type == "cable":
        provider = form_data.get("provider")
        bouquet = form_data.get("bouquet")
        smartcard = form_data.get("smartcard")
        # provider = get_provider_by_id(provider_id)
        # bouquet = get_plan_from_provider(provider, bouquet_id) if provider else None
        order = Order.objects.create(
            ref=order_ref,
            customer=customer,
            service_type="cable",
            network=provider,
            phone_number=phone,
            amount=amount,
        )
        if bouquet:
            order.plan_id =  bouquet
        order.smartcard = smartcard
        order.save()
        create_order_virtual_account(customer, order)  # Create PalmPay virtual account for this order

        payment = Payment.objects.create(order=order, amount=amount,
                              account_reference=f"ORD-{order.id.hex[:10].upper()}")
        order.status = "WAITING_PAYMENT"
        order.save()
        if customer.palmpay_account_number and customer.palmpay_bank_name:
            send_whatsapp_message(
                phone,
                {
                    "type": "text",
                    "text": {
                        "body": f"Please pay ₦{amount} to:\n\nAcct: {customer.palmpay_account_number}\nBank: {customer.palmpay_bank_name}\nRef: {payment.account_reference}\n\nYour service will be delivered after payment is confirmed."
                    }
                }
            )
