# bot/utils.py
from django.conf import settings
import requests
import logging
import json
from django.utils.crypto import get_random_string

from aibot.models import Plan, Provider

WHATSAPP_TOKEN = settings.WHATSAPP_TOKEN
WHATSAPP_PHONE_NUMBER_ID = settings.WHATSAPP_PHONE_NUMBER_ID


def send_whatsapp_message(to_number, message_payload):
    """
    Send a message using Meta WhatsApp Cloud API.
    message_payload should be the JSON body under "interactive" or "text", etc (we build top-level fields here).
    """
    url = f"https://graph.facebook.com/v24.0/{WHATSAPP_PHONE_NUMBER_ID}/messages"
    headers = {
        "Authorization": f"Bearer {WHATSAPP_TOKEN}",
        "Content-Type": "application/json",
        "X-FB-Debug": "true",
    }
    data = {"messaging_product": "whatsapp", "to": to_number}
    data.update(message_payload)
    print("Sending WhatsApp message:", json.dumps(data))
    resp = requests.post(url, json=data, headers=headers, timeout=10)
    if resp.status_code != 200:
        print("\n❌ WHATSAPP API ERROR:", resp.text, "\n")

    resp.raise_for_status()
    return resp.json()


def get_airtime_providers(title):
    """
    Example function to return a list of airtime providers.
    In a real app, this might query a database or external API.
    """
    if title == "mtn":
        return 8
    elif title == "glo":
        return 10
    elif title == "9mobile":
        return 7
    elif title == "airtel":
        return 9
    return None


def swiftlink_login():
    url = "https://swiftlinkng.com/api/login"
    payload = {
        "email": settings.SWIFTLINK_EMAIL,
        "password": settings.SWIFTLINK_PASSWORD,
    }
    headers = {"Content-Type": "application/json"}
    try:
        resp = requests.post(url, json=payload, headers=headers, timeout=10)
        print("Swiftlink login response:", resp.text)
        resp.raise_for_status()
        data = resp.json().get("data", {})
        return data.get("token")
    except Exception as e:
        logging.exception("Swiftlink login error")
        return None


def sync_providers_and_plans(api_data):
    """
    Syncs the Provider and Plan tables with the latest data from get_providers().
    Call this only when you want to refresh the DB cache.
    """
    for item in api_data["data"]:
        provider, _ = Provider.objects.update_or_create(
            subcategory_id=item["subcategory_id"],
            defaults={
                "title": item["title"],
                "category": item["category"],
                "serviceID": item.get("serviceID", ""),
                "status": item.get("status", 1),
            },
        )
        # Remove old plans for this provider
        provider.plans.all().delete()
        for plan in item.get("plan", []):
            Plan.objects.create(
                provider=provider, plan=plan["plan"], amount=plan["amount"]
            )


def get_providers():
    """
    Example function to return a list of data providers.
    In a real app, this might query a database or external API.
    """
    url = "https://swiftlinkng.com/api/get/plans"
    token = swiftlink_login()
    headers = {"Authorization": f"Bearer {token}", "Accept": "application/json"}

    response = requests.get(url, headers=headers)

    print("GETTING PROVIDERS AND PLANS FROM API")

    # print(response.status_code, response.text)

    # Optional: handle errors
    if response.status_code != 200:
        return {
            "error": "Failed to fetch plans",
            "status_code": response.status_code,
            "details": response.text,
        }

    sync_providers_and_plans(response.json())


def get_bouquets(plan):
    """
    Example function to return a list of data providers.
    In a real app, this might query a database or external API.
    """
    url = f"https://swiftlinkng.com/api/fetch/bouquet?plan={plan}"
    token = swiftlink_login()
    headers = {"Authorization": f"Bearer {token}", "Accept": "application/json"}

    print("Fetching bouquets for plan:", plan)
    response = requests.get(url, headers=headers)

    print("GETTING  BOUQUETS FROM API FOR PLAN:", plan)

    # print(response.status_code, response.text)

    # Optional: handle errors
    if response.status_code != 200:
        return {
            "error": "Failed to fetch plans",
            "status_code": response.status_code,
            "details": response.text,
        }

    print("Bouquet response:", response.text)

    return response.json()


def get_provider_by_id(provider_id):
    raw = get_providers()
    for item in raw["data"]:
        if str(item["subcategory_id"]) == str(provider_id):
            return item
    return None


def get_plan_from_provider(provider, plan_id):
    # plan_id is the index you assigned in the flow (string or int)
    idx = int(plan_id)

    plans = provider.get("plan", [])
    if idx < 0 or idx >= len(plans):
        return None

    return plans[idx]


def get_provider_list(category=None):
    """
    Returns a list of providers for dropdowns, etc.
    """
    return [
        {"id": str(p.subcategory_id), "title": p.title}
        for p in Provider.objects.filter(status=1, category=category)
    ]


def get_cable_provider_list(category=None):
    """
    Returns a list of providers for dropdowns, etc.
    """
    return [
        {"id": str(p.title).lower(), "title": p.title}
        for p in Provider.objects.filter(status=1, category=category)
    ]


def build_plan_list(provider_id):
    """
    Returns a list of plans for a given provider.
    """
    try:
        provider = Provider.objects.get(subcategory_id=provider_id)
    except Provider.DoesNotExist:
        return []
    return [
        {
            "id": str(plan.id),
            "title": f"{plan.plan} | ₦{str(float(plan.amount))}",
            "title": f"{plan.plan} | ₦{str(float(plan.amount))}",
        }
        for plan in provider.plans.all()
    ]


def get_amount_value(provider_id):
    try:
        provider = Provider.objects.get(subcategory_id=provider_id)
        amount = Plan.objects.filter(provider=provider).first().amount
        return str(amount)
    except Provider.DoesNotExist:
        return ""


def get_provider_title(provider_id):
    try:
        provider = Provider.objects.get(subcategory_id=provider_id)
        return provider.title
    except Provider.DoesNotExist:
        return ""


def get_cable_provider_title(provider_id):
    try:
        provider = Provider.objects.get(title=provider_id)
        return provider.title
    except Provider.DoesNotExist:
        return ""


def get_plan_title(plan_id):
    try:
        print("Getting plan title for plan_id:", plan_id)
        # provider = Provider.objects.get(subcategory_id=provider_id)
        plan = Plan.objects.get(id=plan_id)
        return plan
    except Plan.DoesNotExist:
        return ""


def build_cable_provider_list(plan):
    print("Getting cable provider list for plan:", plan)
    raw = get_bouquets(plan)
    providers = []
    for item in raw["data"]:
        providers.append({"id": str(item["variation_code"]), "title": item["name"]})

    print("Built cable provider list:", providers)
    return providers


def build_bouquet_provider_list(provider, plan):
    print("Getting cable provider list for plan:", plan)
    raw = get_bouquets(provider)
    for item in raw["data"]:
        if item["variation_code"] == plan:
            return {
                "variation_code": str(item["variation_code"]),
                "variation_amount": item["variation_amount"],
                "name": item["name"],
            }

    # print("Built cable provider list:", providers)
    # return providers


# def get_bouquet_title(plan_id):
#     try:
#         print("Getting plan title for plan_id:", plan_id)
#         # provider = Provider.objects.get(subcategory_id=provider_id)
#         plan = Plan.objects.get(id=plan_id)
#         return plan
#     except Plan.DoesNotExist:
#         return ""


def build_cable_list(provider_id):
    """
    Returns a list of plans for a given provider.
    """
    try:
        provider = Provider.objects.get(subcategory_id=provider_id)
    except Provider.DoesNotExist:
        return []
    return [
        {
            "id": str(plan.id),
            "title": f"{plan.plan} | ₦{str(float(plan.amount))}",
            "title": f"{plan.plan} | ₦{str(float(plan.amount))}",
        }
        for plan in provider.plans.all()
    ]


def build_bouquet_list(provider_id):
    raw = get_bouquets(provider_id)
    bouquets = []
    for item in raw["data"]:
        bouquets.append(
            {
                "id": f"{item['plan_id']}|{item['title']}|{item['amount']}",
                "title": item["title"],
                "amount": item["amount"],
            }
        )
    return bouquets


# def get_provider_title(provider_id):
#     for p in build_provider_list():
#         if p["id"] == str(provider_id):
#             return p["title"]
#     return ""


def get_bouquet_title(provider_id, bouquet_id):
    bouquets = build_bouquet_list(provider_id)
    for b in bouquets:
        if b["id"].startswith(bouquet_id):
            return b["title"]
    return ""


def fulfill_swiftlink_request(order):
    token = swiftlink_login()

    """
    Call Swiftlink API to fulfill the user's order (airtime, data, DSTV, etc.).
    """
    logger = logging.getLogger(__name__)
    # Example: Airtime purchase
    if order.service_type == "airtime":
        url = "https://swiftlinkng.com/api/purchase/airtime"
        payload = {
            "subcategory_id": order.network,
            "phonenumber": order.phone_number,
            "amount": float(order.amount),
            "plan_id": order.plan_id,
            "customer_reference": str(order.id),
        }
        print("Swiftlink login token:", token)
        headers = {
            "Authorization": f"Bearer {token}",
            "Content-Type": "application/json",
        }
        try:
            resp = requests.post(url, json=payload, headers=headers, timeout=15)
            resp.raise_for_status()
            logger.info(f"Swiftlink airtime response: {resp.text}")
            # Notify user after fulfillment
            if order.customer and hasattr(order.customer, "whatsapp_number"):
                try:
                    if order.service_type == "airtime":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "Airtime purchase successful!"},
                            },
                        )
                    elif order.service_type == "data":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "Data bundle purchase successful!"},
                            },
                        )
                    elif order.service_type == "dstv":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "DSTV payment successful!"},
                            },
                        )
                except Exception as e:
                    logger.error(f"Failed to notify user via WhatsApp: {e}")
            return resp.json()
        except Exception as e:
            logger.exception("Swiftlink API error (airtime)")
            if order.customer and hasattr(order.customer, "whatsapp_number"):
                try:
                    send_whatsapp_message(
                        order.customer.whatsapp_number,
                        {
                            "type": "text",
                            "text": {"body": f"Airtime purchase failed: {e}"},
                        },
                    )
                except Exception as notify_err:
                    logger.error(f"Failed to notify user of error: {notify_err}")
            return None
    # Data bundle purchase
    if order.service_type == "data":
        url = "https://swiftlinkng.com/api/purchase/data"
        payload = {
            "subcategory_id": order.network,
            "phonenumber": order.phone_number,
            "plan_id": getattr(order, "plan_id", None),
            "amount": float(order.amount),
            "customer_reference": str(order.id),
        }
        headers = {
            "Authorization": f"Bearer {token}",
            "Content-Type": "application/json",
        }
        try:
            resp = requests.post(url, json=payload, headers=headers, timeout=15)
            resp.raise_for_status()
            logger.info(f"Swiftlink data response: {resp.text}")
            # Notify user after fulfillment
            if order.customer and hasattr(order.customer, "whatsapp_number"):
                try:
                    if order.service_type == "airtime":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "Airtime purchase successful!"},
                            },
                        )
                    elif order.service_type == "data":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "Data bundle purchase successful!"},
                            },
                        )
                    elif order.service_type == "dstv":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "DSTV payment successful!"},
                            },
                        )
                except Exception as e:
                    logger.error(f"Failed to notify user via WhatsApp: {e}")
            return resp.json()
        except Exception as e:
            logger.error(f"Swiftlink API error (data): {e}")
            return None
    # DSTV purchase
    if order.service_type == "cable":
        url = "https://swiftlinkng.com/api/purchase/cable"
        payload = {
            "cardno": order.smartcard,
            "variation_code": order.plan_id,
            "amount": float(order.amount),
            "customer_reference": str(order.id),
            "phonenumber": order.phone_number,
            "plan": order.network,
        }
        headers = {
            "Authorization": f"Bearer {token}",
            "Content-Type": "application/json",
        }
        try:
            resp = requests.post(url, json=payload, headers=headers, timeout=15)
            resp.raise_for_status()
            logger.info(f"Swiftlink dstv response: {resp.text}")
            # Notify user after fulfillment
            if order.customer and hasattr(order.customer, "whatsapp_number"):
                try:
                    if order.service_type == "airtime":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "Airtime purchase successful!"},
                            },
                        )
                    elif order.service_type == "data":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "Data bundle purchase successful!"},
                            },
                        )
                    elif order.service_type == "cable":
                        send_whatsapp_message(
                            order.customer.whatsapp_number,
                            {
                                "type": "text",
                                "text": {"body": "Cable payment successful!"},
                            },
                        )
                except Exception as e:
                    logger.error(f"Failed to notify user via WhatsApp: {e}")
            return resp.json()
        except Exception as e:
            logger.error(f"Swiftlink API error (dstv): {e}")
            return None
    return None


def create_unique_order_ref():
    from .models import Order

    not_unique = True
    while not_unique:
        unique_ref = (
            f"SWL{get_random_string(7, 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789')}"
        )
        if not Order.objects.filter(ref=unique_ref):
            not_unique = False
    return str(unique_ref)
