# bot/views.py
import base64
import json
import logging
import time
from uuid import uuid4
from django.conf import settings
from django.http import JsonResponse, HttpResponse
from django.views.decorators.csrf import csrf_exempt

from aibot.flows.flows import HANDLER_MAP, FLOW_CONFIG, send_airtime_flow, send_data_flow, send_dstv_flow, send_service_buttons
from .models import Customer
from .utils import send_whatsapp_message

logger = logging.getLogger(__name__)

from base64 import b64decode, b64encode
from cryptography.hazmat.primitives.asymmetric.padding import OAEP, MGF1
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.ciphers import algorithms, Cipher, modes
from cryptography.hazmat.primitives.serialization import load_pem_private_key


PRIVATE_KEY =  settings.PRIVATE_KEY  # full PEM string


@csrf_exempt
def whatsapp_flow_endpoint(request):
    """Unified WhatsApp Flow endpoint supporting multiple WhatsApp Flows."""

    print("[FLOW ENDPOINT] Received WhatsApp Flow request")
    print("[FLOW ENDPOINT] Body:", request.body.decode()[:500])  # First 500 chars
    logger.info("[FLOW ENDPOINT] Request received")
    
    if request.method != "POST":
        return HttpResponse("ok")

    try:
        # -----------------------------
        # 1️⃣ Decode & Decrypt Request
        # -----------------------------
        body = json.loads(request.body.decode())
        enc_key = body.get("encrypted_aes_key")
        enc_data = body.get("encrypted_flow_data")
        iv_b64 = body.get("initial_vector")

        data_dict, aes_key, iv = decrypt_request(enc_data, enc_key, iv_b64)
        
        print("[FLOW ENDPOINT] Decrypted Flow Data:", json.dumps(data_dict, indent=2))
        logger.info(f"[FLOW ENDPOINT] Decrypted: {json.dumps(data_dict, indent=2)}")

        action = data_dict.get("action")
        screen = data_dict.get("screen")
        form_data = data_dict.get("data", {})
        flow_token = data_dict.get("flow_token", "")

        print(f"[FLOW ENDPOINT] action={action}, screen={screen}, flow_token={flow_token}")
        logger.info(f"[FLOW ENDPOINT] action={action}, screen={screen}, flow_token={flow_token}")
        logger.info(f"[FLOW ENDPOINT] form_data={form_data}")

        # Helper: uniform encrypted response
        def send(resp_obj):
            encrypted = encrypt_response(resp_obj, aes_key, iv)
            return HttpResponse(encrypted, content_type="text/plain")

        # ---------------------------------------
        # 2️⃣ Healthcheck Ping (Meta internal)
        # ---------------------------------------
        if action == "ping":
            return send({"data": {"status": "active"}})

        # ---------------------------------------
        # 3️⃣ Determine Flow Type from Token
        # ---------------------------------------
        flow_type = next(
            (key for key in FLOW_CONFIG.keys() if flow_token.startswith(f"{key}_")),
            None,
        )

        if not flow_type:
            logger.warning(f"❌ Unknown flow token prefix → {flow_token}")
            return send(
                {"screen": screen, "data": {"error_message": "Unknown flow token"}}
            )

        flow_conf = FLOW_CONFIG[flow_type]
        handler_func = HANDLER_MAP.get(flow_conf["handler"])

        if not handler_func:
            logger.error(f"❌ No handler found for flow {flow_type}")
            return send(
                {"screen": screen, "data": {"error_message": "Handler missing"}}
            )

        logger.info(
            f"🔀 Resolved → flow_type={flow_type}, handler={handler_func.__name__}"
        )

        # ---------------------------------------
        # 4️⃣ INIT (Flow Started)
        # ---------------------------------------
        if action == "INIT":
            print("[FLOW ENDPOINT] ✅ INIT ACTION TRIGGERED")
            initial = flow_conf["initial_screen"]
            logger.info(f"[FLOW ENDPOINT] 🚀 INIT → starting at {initial}")
            print(f"[FLOW ENDPOINT] Calling handler with initial screen: {initial}")
            response = handler_func(form_data, flow_token, initial)
            print(f"[FLOW ENDPOINT] Handler returned: {json.dumps(response, indent=2)}")
            return send(response)

        # ---------------------------------------
        # 5️⃣ data_exchange or navigate (normal screens)
        # ---------------------------------------
        is_confirm_screen = screen == flow_conf["confirm_screen"]

        # (A) Normal flow steps BEFORE confirm screen
        if action in ("data_exchange", "navigate") and not is_confirm_screen:
            logger.info(f"➡️ data_exchange BEFORE confirm → {screen}")
            response = handler_func(form_data, flow_token, screen)
            return send(response)

        # (B) confirm screen → final processing
        if action == "data_exchange" and is_confirm_screen:
            logger.info("🧾 CONFIRM SCREEN SUBMISSION")
            response = handler_func(form_data, flow_token, screen)
            return send(response)

        # ---------------------------------------
        # 6️⃣ Back Action
        # ---------------------------------------
        if action == "back":
            initial = flow_conf["initial_screen"]
            logger.info(f"↩️ BACK → {initial}")
            return send({"screen": initial, "data": {}})

        # ---------------------------------------
        # 7️⃣ Unknown Actions
        # ---------------------------------------
        logger.warning(f"⚠️ Unsupported action: {action}")
        return send(
            {
                "screen": screen,
                "data": {"error_message": f"Unsupported action: {action}"},
            }
        )

    except Exception as e:
        logger.exception("❌ FLOW PROCESSING ERROR")
        error_b64 = base64.b64encode(f"Error: {e}".encode()).decode()
        return HttpResponse(error_b64, content_type="text/plain", status=500)


def decrypt_request(enc_flow_b64, enc_key_b64, iv_b64):
    """Decrypt WhatsApp Flow payload using RSA-OAEP-SHA256 and AES-GCM"""

    encrypted_flow = b64decode(enc_flow_b64)
    encrypted_aes_key = b64decode(enc_key_b64)
    iv = b64decode(iv_b64)
    
    key_bytes = base64.b64decode(PRIVATE_KEY)
    private_key = load_pem_private_key(key_bytes, password=None)

    # private_key = load_pem_private_key(PRIVATE_KEY.encode(), password=None)

    # ✅ RSA OAEP SHA-256 — MUST MATCH Meta
    aes_key = private_key.decrypt(
        encrypted_aes_key,
        OAEP(
            mgf=MGF1(algorithm=hashes.SHA256()),
            algorithm=hashes.SHA256(),
            label=None
        )
    )

    # ✅ Split ciphertext and tag
    ciphertext = encrypted_flow[:-16]
    tag = encrypted_flow[-16:]

    decryptor = Cipher(
        algorithms.AES(aes_key),
        modes.GCM(iv, tag)
    ).decryptor()

    decrypted_bytes = decryptor.update(ciphertext) + decryptor.finalize()

    return json.loads(decrypted_bytes.decode()), aes_key, iv


def encrypt_response(response, aes_key, iv):
    # Flip the initialization vector
    flipped_iv = bytearray()
    for byte in iv:
        flipped_iv.append(byte ^ 0xFF)

    # Encrypt the response data
    encryptor = Cipher(algorithms.AES(aes_key),
                       modes.GCM(flipped_iv)).encryptor()
    return b64encode(
        encryptor.update(json.dumps(response).encode("utf-8")) +
        encryptor.finalize() +
        encryptor.tag
    ).decode("utf-8")


from django.core.cache import cache

def recently_sent_flow(phone):
    key = f"flow_sent:{phone}"
    return cache.get(key)

def mark_flow_sent(phone):
    key = f"flow_sent:{phone}"
    cache.set(key, "1", 60 * 60)   # 1 hour TTL


@csrf_exempt
def whatsapp_webhook(request):
    """Handles inbound text + button clicks and triggers WhatsApp Flow."""
    # GET: verification
    
    print("webhook is being accessed")
    
    if request.method == "GET":
        mode = request.GET.get("hub.mode")
        token = request.GET.get("hub.verify_token")
        challenge = request.GET.get("hub.challenge")
        if mode and token:
            if token == settings.VERIFY_TOKEN:
                return HttpResponse(challenge)
            return HttpResponse("Verification token mismatch", status=403)
        return HttpResponse("OK")

    # POST: events
    try:
        body = json.loads(request.body.decode("utf-8"))
        # print(body)
        # logger.info("Incoming WhatsApp webhook: %s", json.dumps(body, indent=2))
    except json.JSONDecodeError:
        return JsonResponse({"ok": False, "error": "Invalid JSON"}, status=400)

    try:
        entry = body.get("entry", [])
        if not entry:
            return JsonResponse({"ok": True, "reason": "no_entry"})
        change = entry[0].get("changes", [{}])[0]
        value = change.get("value", {})
        if not value:
            return JsonResponse({"ok": True, "reason": "no_value"})

        # Delivery / status updates
        if "statuses" in value:
            logger.info("Status update: %s", value["statuses"])
            return JsonResponse({"ok": True, "type": "status"})

        messages = value.get("messages", [])
        if not messages:
            return JsonResponse({"ok": True, "reason": "no_messages"})

        msg = messages[0]
        from_number = msg.get("from")
        msg_type = msg.get("type")

        # Ensure a customer record exists (kept simple)
        # customer, _ = Customer.objects.get_or_create(whatsapp_number=from_number)

        # 1) Plain text -> show menu / trigger flow
        if msg_type == "text":
            text = (msg.get("text", {}).get("body") or "").strip().lower()
            if text in ("airtime", "buy airtime"):
                if recently_sent_flow(from_number):
                    logger.info("Ignoring repeat flow trigger for %s", from_number)
                    return JsonResponse({"ok": True})
                flow_token = f"buy_airtime_v2_{uuid4().hex}"
                send_airtime_flow(from_number, flow_token, mode="published")
                mark_flow_sent(from_number)
                return JsonResponse({"ok": True})
            if text in ("cable", "buy cable"):
                if recently_sent_flow(from_number):
                    logger.info("Ignoring repeat flow trigger for %s", from_number)
                    return JsonResponse({"ok": True})
                flow_token = f"dstv_purchase_v2_{uuid4().hex}"
                send_dstv_flow(from_number, flow_token, mode="published")
                mark_flow_sent(from_number)
                return JsonResponse({"ok": True})
            if text in ("data", "buy data"):
                if recently_sent_flow(from_number):
                    logger.info("Ignoring repeat flow trigger for %s", from_number)
                    return JsonResponse({"ok": True})
                flow_token = f"data_purchase_v2_{uuid4().hex}"
                send_data_flow(from_number, flow_token, mode="published")
                mark_flow_sent(from_number)
                return JsonResponse({"ok": True})
            elif text in ("menu", "start", "hello", "hi", "help", "good morning", "good afternoon", "good evening", "I want"):
                name = value.get("contacts", [{}])[0].get("profile", {}).get("name", "")
                send_service_buttons(from_number, name)
                return JsonResponse({"ok": True})

        # 2) Modern button reply (interactive.button_reply.id)
        if (
            msg_type == "interactive"
            and msg.get("interactive", {}).get("type") == "button_reply"
        ):
            btn_id = msg["interactive"]["button_reply"]["id"]
            if btn_id == "airtime_AIRTIME_FORM":
                flow_token = f"buy_airtime_v2_{uuid4().hex}"
                send_airtime_flow(from_number, flow_token, mode="published")
            elif btn_id == "data_DATA_FORM":
                flow_token = f"data_purchase_v2_{uuid4().hex}"
                send_data_flow(from_number, flow_token, mode="published")
            elif btn_id == "dstv_DSTV_FORM":
                flow_token = f"dstv_purchase_v2_{uuid4().hex}"
                send_dstv_flow(from_number, flow_token, mode="published")
            elif btn_id == "contact_support":
                send_whatsapp_message(
                    from_number,
                    {
                        "type": "text",
                        "text": {
                            "body": (
                                "📞 *Swiftlink Support*\n\n"
                                "For assistance, please reach out to us:\n"
                                "📧 Email: support@swiftlinkng.com\n"
                                "📱 WhatsApp: +2349000000000\n\n"
                                "We typically respond within a few minutes."
                            )
                        },
                    },
                )
            else:
                send_whatsapp_message(
                    from_number, {"type": "text", "text": {"body": "Unknown option."}}
                )
            return JsonResponse({"ok": True})

        # 3) Legacy button (older payload style) – keep for backward compatibility
        if msg_type == "button":
            payload_id = msg.get("button", {}).get("payload")
            if payload_id == "airtime_AIRTIME_FORM":
                flow_token = f"buy_airtime_v2_{uuid4().hex}"
                send_airtime_flow(from_number, flow_token, mode="published")
            elif payload_id == "data_DATA_FORM":
                flow_token = f"data_purchase_v2_{uuid4().hex}"
                send_data_flow(
                    from_number,
                    flow_token=flow_token,
                    mode="published",
                )
            elif payload_id == "dstv_DSTV_FORM":
                flow_token = f"dstv_purchase_v2_{uuid4().hex}"
                send_dstv_flow(
                    from_number,
                    flow_token=flow_token,
                    mode="published",
                )
            else:
                send_whatsapp_message(
                    from_number, {"type": "text", "text": {"body": "Unknown option."}}
                )
            return JsonResponse({"ok": True})

        # NOTE: Flow submissions DO NOT arrive here for Flows.
        # They go to your Flow Endpoint (AES-GCM) that you already implemented.
        return JsonResponse({"ok": True, "reason": f"unhandled_type:{msg_type}"})

    except Exception as e:
        logger.exception("Error in WhatsApp webhook")
        return JsonResponse({"ok": False, "error": str(e)}, status=500)
