import json
import logging
from django.views.decorators.csrf import csrf_exempt
from django.http import JsonResponse, HttpResponse
from django.utils import timezone
from aibot.models import Customer, Payment
from aibot.services.palmpay import verify_palmpay_webhook_signature

logger = logging.getLogger(__name__)

@csrf_exempt
def palmpay_webhook(request):
    if request.method != "POST":
        return HttpResponse("ok")
    if not verify_palmpay_webhook_signature(request):
        return JsonResponse({"ok": False, "error": "Invalid signature"}, status=403)
    try:
        data = json.loads(request.body.decode())
        event = data.get("event")
        payload = data.get("data", {})
        if event == "payment.success":
            account_ref = payload.get("accountReference")
            amount = payload.get("amount")
            transaction_id = payload.get("transactionId")
            customer = Customer.objects.filter(palmpay_account_ref=account_ref).first()
            if not customer:
                logger.warning(f"PalmPay: No customer for ref {account_ref}")
                return JsonResponse({"ok": False, "error": "Customer not found"}, status=404)
            payment = Payment.objects.filter(order__customer=customer, status="PENDING").last()
            if not payment:
                logger.warning(f"PalmPay: No pending payment for customer {customer.id}")
                return JsonResponse({"ok": False, "error": "No pending payment"}, status=404)
            payment.status = "SUCCESS"
            payment.palmpay_transaction_id = transaction_id
            payment.date_paid = timezone.now()
            payment.save()
            payment.order.status = "PAID"
            payment.order.save()
            # Trigger Celery fulfillment task
            from aibot.tasks import send_airtime_task, send_data_task, send_dstv_task
            if payment.order.service_type == "airtime":
                send_airtime_task.delay(payment.order.id)
            elif payment.order.service_type == "data":
                send_data_task.delay(payment.order.id)
            elif payment.order.service_type == "cable":
                send_dstv_task.delay(payment.order.id)
            return JsonResponse({"ok": True})
        return JsonResponse({"ok": True, "reason": "unhandled_event"})
    except Exception as e:
        logger.exception("PalmPay webhook error")
        return JsonResponse({"ok": False, "error": str(e)}, status=500)
